/*	  ========================================================================================== 
 *     A Btree node should know its left most, right most and parent node.
 *     It should know its value and which level it lies on within the tree.
 * 
 */
class Node{

	/**
	 * Create a Node for the binary tree
	 * @param  {Node} left  the node on the left of this node; a child
	 * @param  {Node} right the node on the right of this node; a child
	 * @param  {Node} the parent node
	 * @return {void}       nothing
	 */
	constructor(left, right,height,value, x, y){
		this.left = left;
		this.right = right;
		this.height = height;
		this.value = value;
		this.x = x;
		this.y = y;
	}
   /**
    * This function updates the height of the Node 
    * * @return {[void]}            none
    */
	updateHeight()
	{
       // add to the greatest height of the current node or branches height.
   	 var l = this.left == null ? -1 : this.left.height;
   	 var r = this.right == null ? -1 : this.right.height;  	 
   	 this.height =  ( l > r ? l : r ) + 1 ;
	
	}
   /**
    * Render the node. 
    * @param  {[canvas]} the canvas object, on which; the node shall be drawn.
    * @return {[void]}             none
    */
	render(ctx)
	{
		var radius = 25;

		ctx.beginPath();
		ctx.arc(this.x, this.y,radius, 0, 2* Math.PI,false );
		ctx.fillStyle = '#fff';
		ctx.fill();
		ctx.lineWidth = 1;
		ctx.strokeStyle = 'MediumBlue';
		ctx.stroke();
		ctx.textBaseline="top";
		ctx.fillStyle="MediumBlue";
		ctx.font = "15px serif";
		ctx.fillText(this.value,this.x -10 , this.y - 10);
	}
};

/**
 *
 *     Tree Class  - this class represents a BTree 
 *	  ========================================================================================== 
 *     A Btree populates itself recursively by popping or pulling values.
 *     When the tree becomes unbalanced it re-organizes itself to rebalance its branches.
 * 
 */
class Tree{

   /**
    * create the tree object 
    * @param  {[type]} screenWidth [description]
    * @return {[type]}             [description]
    */
   constructor(){
   	
   		this.root = null;
   		this.treeJSON = {};
   		this.depthLeft = 0;
   		this.depthRight = 0;
   		this.canvas = document.getElementById('btree');
   		this.ctx = this.canvas.getContext("2d");
      this.size = 0;

   }

   /**
    * Public function
    * Inserts a value into the AVL tree.
    */
   insert(value)
   {
   	 var val = Number(value);
   	 this.root = this.push( this.root, val);
   }
   /**
    * Finds a value and alerts when found.
    * @param {number} Value what we seek.
    * @return {Void} none.
    */
   seek(value)
   {
      var val = Number(value);
      this.root = this.find(this.root,val);
   }
   /**
    * seeks helper function to find the value in question.
    * @param {node} node the node in current traversal.
    * @param {number} value the value we seek.
    * @return {node} the current node.
    */
   find(node,value)
   {
      //found our node..print it.
      if (node.value == value)
      {
         alert("Found value: " + value);
      }
      // doesnt exist.
      else if (node.left == null && node.right == null){
         alert("Oops that value: " + value +" doesn't exist!");
         return node;
      }
      // keep traversing 
      // the value we seek is to right of the branch since it is greater than this nodes value.
      else if (value > node.value)
         {
            node.right = this.find(node.right,value);

        }
      // keep traversing 
      // the value we seek is to left of the branch since it is lesser than this nodes value.
         else if ( value < node.value)
         {
            node.left = this.find(node.left,value);

         }

         return node

   }

   /**
    * Deletes a value from the AVL tree
    * @param  {Number} value the number inserted into the tree
    * @return {Void}       none
    */
   delete(value){
   	 var val = Number(value);
   	 this.root = this.remove(this.root,val );
   }
   /**
    * Prints the tree.
    * @return {Void}       none
    */
   print()
   {
      // the json file is there should the tree ever need to be exported, meh..
   	this.treeJSON = {};
      // clear the canvas for a new drawing...
   	this.ctx.clearRect(0,0,this.canvas.width,this.canvas.height);
      
      // set the roots x and y coordinate..
      // note the canvas origin is as follows:
      //  _____________
      // | (0,0)
      // |

      this.root.x = this.canvas.width/2;
      this.root.y = 30;

   	this.root =this.crawl(this.root, this.root.x / 2 ); 
   }
   /**
    * Hey lines are nice, why not, lets represent the connections to the node.
    * draw the line from the parent to the child.
    * @param  {parent} the parent node.
    * @param  {child} the child node.
    * @return {Void}       none
    */
   drawLine(parent,child )
   {
      // what better color than dark see green.
   	this.ctx.strokeStyle = 'DarkSeaGreen';
   	if (parent != null && child != null )
   	{
   		this.ctx.beginPath();
   		this.ctx.moveTo(parent.x,parent.y);
   		this.ctx.lineTo(child.x,child.y);
   		this.ctx.stroke();
   	}


   }
   /**
    * Uses recursion to traverse the tree, calls itself until the node is either inserted, found or deleted then unwraps 
    * upon itself back to the root.
    * @param  {node} the parent node, alas a beginning anew.
    * @param  {xinterval} as the trees branches grow for spaces sake, and aesthetics considered the 
    * distance between the nodes at their level will decrease as their height decreases.
    * @return {Void}       none
    */
   crawl(node , xinterval )
   {
   		var level = node.height.toString();
   		var yinterval = (this.canvas.height-55)/(this.root.height + 1);
   		var ht = node.height == 0 ? 1 : node.height;
   		
         // we are not yet where we need to be go leftwards.
   		if(node.left != null)
   		{
            // perform spacing...
   			node.left.y = node.y + yinterval + 5;
   			node.left.x = node.x - xinterval;

            // draw the line...
   			this.drawLine(node,node.left);

   			this.crawl(node.left, xinterval / 2);
   			
   		}
         // starboard side until we insert at the leaf..
   		if(node.right != null)
   		{
            //space out
   			node.right.y = node.y +  yinterval;
   			node.right.x = node.x  + xinterval-5 ;

            //draw the lines and continue.
   			this.drawLine(node,node.right);
   			this.crawl(node.right, xinterval / 2);
   		}
   
      // write to our json should one day, perhaps, maybe , just might happen on the occurence that on chance  we require to export the tree.
   		if (level in this.treeJSON)
   		{
   			this.treeJSON[level].push(node.value.toString());
         }
   		else{

   			this.treeJSON[level] = [node.value.toString()];

   		}
       // draw the node...
		 node.render(this.ctx);	
		 return node;
   	
   }
   /**
    * If either side of the current branch or node is greater by more than 1 then the tree is unbalanced
    * and some manipulations are in order.
    * @param  {lt} the left side.
    * @param  {rt} the right side.
    * @return {Void}       none
    */
   unbalanced(lt,rt){
      // left (lt) or right (rt) are arbitrary
      // this function could be called at unbalanced(right,left)
      // and lt would really be right.
   	var l = lt == null ? -1 : lt.height;
   	var r = rt == null ? -1 : rt.height;

      // if the difference exceeds 1 it is unbalanced.
      return l - r > 1 ? true : false ;
   }

   /**
    * Places a node on the AVL tree as a leaf and then rebalances the tree should the need arise.
    * @param  {node} the current node on the traversal to the leafy greens, i.e. terminal point
    *                where the new node  is placed.
    * @param  {value} the value to be placed to the node.
    * @return {Node}  the node at the current traversal point.
    */
   push( node, value)
   { 
         // hey once the node is null meaning that either right or left is a terminal 
         // point we can create the node and insert it.
   		if(node == null){
   			node =   new Node(null,null,-1,value, 0,0);
        this.size++;
   		 }
          // not there yet if the value exceeds the current node go right.
   		else if (value > node.value)
   		{
            // go right.
   			node.right = this.push(node.right,value);

            // as we unwind the stack observe the trees balance and correct it.
   			if (this.unbalanced(node.right, node.left))
   			{
               // if we were traveling right all along but the leaf was placed to the left.
               // we must first rotate the the child with its children.
               // i.e.               then we rotate again to lessen the height.
               // 2              2
               //  \              \
               //  10 to ...       8   then ..            8
               //  /                \                    / \
               // 8                 10                  2   10

               // but if the leaf was placed to the left 
               // from the steps above only the 2nd and 3rd are necessary.

   				node = ( value < node.right.value) ? this.doubleRotateLeft(node) : this.rotateLeft(node);
   			}
   		}
         // not there yet if the value exceeds the current node go left.
   		else if ( value < node.value)
   		{
   			node.left = this.push(node.left,value);

            // this is the reverse of above when unbalanced.
            // this time if going left then leaf at right correct with child then rotate else just rotate.
   			if (this.unbalanced(node.left, node.right))
   			{
   				node = ( value > node.left.value ) ? this.doubleRotateRight(node) : this.rotateRight(node);
   			}
   		}
         //update the heights.
   		node.updateHeight();
   		return node;
   }

   /**
    * Removes a node from the tree.
    * @param  {node} the current node on the traversal to the leafy greens, i.e. terminal point
    *                where the damned node resides, poor soul.
    * @param  {value} the value to be deleted.
    * @return {Node}  the node at the current traversal point.
    */
   remove( node, value)
   { 
   	// the node to die has been found.
      if( value == node.value)
		{
         // pluck analyzes how we are to remove the node, 
         // to keep the integrity of the tree.
         this.size--;
			return this.pluck(node);
		}
      // didn't find our value.
      else if (node.left == null && node.right == null){
         alert("Oops that value doesn't: "+value+" exist!");
         return node;
      }
   	// keep traversing 
      // the value we seek is to right of the branch since it is greater than this nodes value.
      else if (value > node.value)
   		{
   			node.right = this.remove(node.right,value);

            // as in the placement of the node we balance.
   			if (this.unbalanced(node.right, node.left))
   			{
   				node = ( value < node.right.value) ? this.doubleRotateLeft(node) : this.rotateLeft(node);
   			}
   	  }
      // keep traversing 
      // the value we seek is to left of the branch since it is lesser than this nodes value.
   		else if ( value < node.value)
   		{
   			node.left = this.remove(node.left,value);

   			if (this.unbalanced(node.left, node.right))
   			{
   				node = ( value > node.left.value ) ? this.doubleRotateRight(node) : this.rotateRight(node);
   			}
   		}

   		node.updateHeight();
   		return node;
   }

   /**
    * The pluck method does the heavy lifting for the remove function.
    * If the node is a leaf return null to destroy it,
    * If it has only one child attach its child to the its parent.
    * But if it has two children then:
    *      when traveling left we get the greatest child and transpose  with the condemned node.
    *      when traveling right we get the least child and transpose  with the condemned node.
    * @param  {node} the current node on the traversal to the leafy greens, i.e. terminal point
    *                where the damned node resides, poor fellow.
    * @return {Node}  the node at the current traversal point.
    */
   pluck(node)
   {
        //case 1: this node is infertile and without child, kill it.
   	  if (node.left == null && node.right == null){
   	  	return  null;
   	  }

        // case 3: this node has a two children on both sides, therefore 
        // if the height of his left child is greater than the right child, we go leftwards and 
        // raise its left child's greatest progeny to its place, behold the king.
        // If the the height of the right child is greater than its leftwards sibling,
        // then we go right but raise its lesser progeny to glory.
   	  if (node.left != null && node.right != null )
   	  {
   	  		if (node.left.height > node.right.height)
   	  		{
   	  			// we go left 
   	  			// if the left node has no child to his right just reroute ..
   	  			if ( node.left.right == null)
   	  			{
   	  				node.left.right = node.right;
   	  				return node.left;
   	  			}
					//else traverse the leaf..
   	  			else{
					node.left = this.leaf(node.left,node,false);
   	  			}

   	  		}
   	  		else
   	  		{
   	  			// we go right 
   	  			// if the right node has no child to his left just reroute ..
   	  			if ( node.right.left == null)
   	  			{
   	  				node.right.left  = node.left;
   	  				return node.right;
   	  			}
					//else traverse the leaf..
   	  			else{
					node.right = this.leaf(node.right,node,true);
   	  			}
   	  		}

   	  		node.updateHeight();
   	  		return node;
   	  }

        // case 2 : only one child, raise its only child to the throne of princehood.
        // in other words the parents pointer pointing to the deleted node will now point to 
        // the nodes child, allowing this node to die quietly and peacefully.
        // -----{--@  may it rest in peace.

   	  node = node.left == null ? node.right : node.left;
	     node.updateHeight();
   	  return node;
   	  
   }
   /**
    * The leaf method is called from the pluck method, 
    * Once a direction in the pluck method is chosed if right,
    * then the leaf will transpose the value and set deleted nodes value to the lesser if right or greater if left child value.
    * @param  {node} the current node on the traversal to the leafy greens, i.e. terminal point
    *                where the damned node resides, poor fellow.
    * @param  {parent} the node to be deleted, its value is altered with the appropriate leaf node value.
    * @param  {left} true if we are required to go leftwards, false otherwise.
    * @return {Node}  the node at the current traversal point.
    */
   leaf(node,parent, left)
   {
   	var value = 0;

      // go leftwards if left or right until traversal leads to the greatest or least child respectively.
      // once found transpose values and set the leaf to null.
      if (left)
      {
      		if (node.left != null) node.left = this.leaf(node.left, parent, true);
      		else
            {
      			parent.value = node.value;
      			node = null;	
      		}
      }
      else{
      		if (node.right != null) node.right = this.leaf(node.right, parent, false);
      		else{
      
      			parent.value = node.value;
      			node = null;
      		}
      }
      return node;
   }

   /**
    * The double rotate right first rotates left with the child of the node and then rotates right with the node.
    * This balances the tree when the traversal was towards the right but the value was on the left subtree of the child.
    *        20                 20
    *       /                  /                12
    *      8      =>         12         =>     / \
    *       \               /                 8  20
    *        12            8     
    *
    * @param  {node} the current node on the traversal to the leafy greens, i.e. terminal point
    *                where the damned node resides, poor fellow.
    * @return {Node}  the node at the current traversal point.
    */
   doubleRotateRight(node)
   {
   	 	node.left = this.rotateLeft(node.left);
   		return this.rotateRight(node);
   }

   /**
    * The double rotate left first rotates right with the child of the node and then rotates left with the node.
    * This balances the tree when the traversal was towards the left but the value was on the right subtree of the child.
    *     5                 5
    *      \                 \                   8
    *      12      =>         8         =>     /  \
    *     /                    \             5     12
    *    8                      12     
    *
    * @param  {node} the current node on the traversal to the leafy greens, i.e. terminal point
    *                where the current node resides.
    * @return {Node}  the node at the current traversal point.
    */
   doubleRotateLeft(node)
   {
   		node.right = this.rotateRight(node.right);
   		return this.rotateLeft(node);
   }

   /**
    * The rotate right occurs when the height of the left subtree exceeds the right subtree by 2 or more.
    *      12
    *     /                 8
    *    8         =>     /  \
    *   /               5     12
    *  5       
    *
     * @param  {node} the current node on the traversal to the leafy greens, i.e. terminal point
    *                where the current node resides.
    * @return {Node}  the node at the current traversal point.
    */
   rotateRight(node){ 
   
      // get the right node i.e. 8 which is the child.
      // set the set the pointer of the left of the node to point to the childs right child.
      // set the childs right to point to our node.
      var child = node.left;
      node.left = child.right == null? null : child.right;
   	child.right = node;
   	  

   	node.updateHeight();
   	child.updateHeight();

      //return the child.
      return child;

   }
   /**
    * The rotate left occurs when the height of the right subtree exceeds the left subtree by 2 or more.
    *      5
    *       \                   8
    *        8         =>     /  \
    *         \             5     12
    *          12     
    *
     * @param  {node} the current node on the traversal to the leafy greens, i.e. terminal point
    *                where the damned node resides, poor fellow.
    * @return {Node}  the node at the current traversal point.
    */
   rotateLeft(node)
    {
        // get the right node i.e. 8 in the figure which is the child.
        // set the set the pointer of the right of the node to point to the childs left child.
        // set the childs left to point to our node.
   	  var child = node.right;
   	  node.right = child.left == null? null : child.left;
   	  child.left = node;
   	  
        // update all heights 
   	  node.updateHeight();
   	  child.updateHeight();

      // return the child.
      return child;
   }



}



